//
// MEV Terminal Program
//
// Version 1.01
//
// PROJECT DESCRIPTION
//
// Kylix Comms Example using Linux Comms I/F
//
// This source code is supplied as an example of a aerial communications
// in Kylix for the Amplicon range of serial cards.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version
// 2 of the License, or (at your option) any later version.
// Although the best endevours have been made to ensure the accuracy
// of this code, MEV make no warranty of any kind with regard to this
// sample source code.
//
// by MEV Ltd
// Copyright (c) 1997-2000 MEV Limited
// Suite 4 Baxall Business Centre, Unit 25b,
// Adswood Road Ind Est, STOCKPORT, Cheshire. SK3 8LF
// +44 (0)161 477 1898
// @mev-ltd.demon.co.uk
//
// ENVIRONMENT
//    Compiler:      Borland KyLix Professional Version 1.0
//    Target system: PC, running Linux
//
// AMENDMENT RECORD
//
// started    1.01  27/09/2001
//
// KNOWN LIMITATIONS / PROBLEMS
//
// Can't paste more that 64K into out going buffer
// A single line can't be more than 1024K
//
// To run this code the executeable must belong to the uucp group and
// have system priveleges. At a su shell prompt in the project directory
// type. (NB you must be super-user or root)
//
// chgrp uucp MEVTerminal
// chmod g+s MEVTerminal
//

unit UnitTerm;

interface

uses
  SysUtils, Types, Classes, Variants, QGraphics, QControls, QForms, QDialogs,
  QExtCtrls, QImgList, QMenus, QTypes, QStdCtrls, QComCtrls,
  unitAbout, libc, unitSerial, QPrinters;

type
  //
  // possible connection states
  //
  TUIState = (itNotConnected, itConnected, itDisConnected);

  TFormTerm = class(TForm)
    MainMenu1: TMainMenu;
    Edit1: TMenuItem;
    Copy1: TMenuItem;
    Paste1: TMenuItem;
    SelectAll1: TMenuItem;
    N1: TMenuItem;
    ClearBuffer1: TMenuItem;
    Transffers1: TMenuItem;
    Help1: TMenuItem;
    NewConnection1: TMenuItem;
    N2: TMenuItem;
    Exit1: TMenuItem;
    SendTextFile1: TMenuItem;
    ViewTextFile1: TMenuItem;
    Contents1: TMenuItem;
    Searchforhelpon1: TMenuItem;
    Howtousehelp1: TMenuItem;
    N4: TMenuItem;
    AboutMEVTerminal1: TMenuItem;
    N3: TMenuItem;
    Settings2: TMenuItem;
    Connect1: TMenuItem;
    Disconnect1: TMenuItem;
    ToolBar1: TToolBar;
    TBNewConnection: TToolButton;
    PopupComPorts: TPopupMenu;
    None1: TMenuItem;
    Com11: TMenuItem;
    Com21: TMenuItem;
    Com31: TMenuItem;
    Com41: TMenuItem;
    Com51: TMenuItem;
    Com61: TMenuItem;
    Com71: TMenuItem;
    Com81: TMenuItem;
    Other1: TMenuItem;
    ToolButton1: TToolButton;
    ToolButton4: TToolButton;
    ToolButton5: TToolButton;
    ToolButton11: TToolButton;
    ImageList1: TImageList;
    TBConnect: TToolButton;
    TBDisconnect: TToolButton;
    TBSettings: TToolButton;
    TBCopy: TToolButton;
    TBPaste: TToolButton;
    TBSelectall: TToolButton;
    TBClearBuffer: TToolButton;
    Panel1: TPanel;
    PanelStatus: TPanel;
    Panel3: TPanel;
    PanelRxStatus: TPanel;
    PanelTxStatus: TPanel;
    Timer1: TTimer;
    N5: TMenuItem;
    Print1: TMenuItem;
    S1: TMenuItem;
    Format1: TMenuItem;
    Font1: TMenuItem;
    FontDialog1: TFontDialog;
    OpenDialog1: TOpenDialog;
    Memo1: TMemo;
    Memo2: TMemo;
    SaveDialog1: TSaveDialog;
    Settings1: TMenuItem;
    Amplicon1: TMenuItem;
    procedure Exit1Click(Sender: TObject);
    procedure ClearBuffer1Click(Sender: TObject);
    procedure NewConnection1Click(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure Disconnect1Click(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure TBNewConnectionClick(Sender: TObject);
    procedure FormKeyPress(Sender: TObject; var Key: Char);
    procedure Timer1Timer(Sender: TObject);
    procedure Copy1Click(Sender: TObject);
    procedure SelectAll1Click(Sender: TObject);
    procedure Paste1Click(Sender: TObject);
    procedure Settings2Click(Sender: TObject);
    procedure Connect1Click(Sender: TObject);
    procedure Print1Click(Sender: TObject);
    procedure S1Click(Sender: TObject);
    procedure AboutMEVTerminal1Click(Sender: TObject);
    procedure Font1Click(Sender: TObject);
    procedure SendTextFile1Click(Sender: TObject);
    procedure ViewTextFile1Click(Sender: TObject);
    procedure Amplicon1Click(Sender: TObject);
  private
    { Private declarations }
    ttyn: string;                      // device name
    hComm  :Integer;                   // comms handle
    lockfile : string;                 // lock file name
    BytesRxed : cardinal;              // byte counts
    BytesTxed : integer;
    RxBuffer: array [0..1023] of char; // TX / RX buffers
    TxBuffer: array [0..1023] of char;

    fTxing:Boolean;                    // flag saying we are transmitting
    fWaitingRx:boolean;                // flag saying we are waiting for rx data
    InTimer:Boolean;                   // re-entrancy flag for timer
    InPaste:Boolean;                   // re-entrancy flag for paste operations
    extra:string;                      // temp buffer for keypresses
    p:integer;                         // position in string for incomomg text

    function GetWantedComProperties(newttyn:string; var UserComParms:TUserComParms):boolean;
    function Connect(newttyn:string;  UserComParms:TUserComParms; var error : string):boolean;
    procedure SetUI(State : TUIState);
    function AreDisConnected:Boolean;
    procedure UpdateMemo(a : array of char; l : cardinal);
    procedure InitComms;
    procedure FreeComms;
  public
    { Public declarations }
    UsercomParms: TUserComParms;       // com port settings
    ttybase : string;                  // default beginning to connection name tty ot ttyA
  end;

const
  INVALID_HANDLE_VALUE = 0;
  MsgFailCon = 'Failed to connect to ';
  
var
  FormTerm: TFormTerm;

implementation

uses UnitConnect, Unitsettings;

{$R *.xfm}

//-------------------------------------------------------------------------
// FormCreate
//-------------------------------------------------------------------------
// Function
//    initialise global vars etc
// Parameters
//    Sender - mandatory (unused)
// Returns
//    None
// Coded
//    helen
//-------------------------------------------------------------------------
procedure TFormTerm.FormCreate(Sender: TObject);
begin
    hComm :=  INVALID_HANDLE_VALUE;

    // find user and pid for this session
    if not InitialiseProcessIDs() then
    begin
      MessageDlg( 'ERROR - unknown user'
                , 'Unable to find you as a user on this system - access to this program is prohibited'
                , mtError
                , [mbok]
                , 0);
      Application.Terminate;
      exit;
    end;
    
    // default settings
    UserComParms.fAmplicon := true;
    UserComParms.baud := 9600;
    UserComParms.bits := 8;
    UserComParms.stopbits := 1;
    UserComParms.parity := 'N';
    UserComParms.fhwflow := false;
    UserComParms.fswflow := false;
    
    ttybase := 'ttyA';
    ttyn := ''; 
    
    // zero byte counts
    BytesTxed := 0;
    BytesRxed := 0;

    // set up flags
    fTxing := false;
    fWaitingRx := false;
    SetUI(itNotConnected);

end;

//-------------------------------------------------------------------------
// Exit
//-------------------------------------------------------------------------
// Function
//    Close Program
// Parameters
//    Sender - manadory (unused)
// Returns
//    None
// Coded
//    helen
//-------------------------------------------------------------------------
procedure TFormTerm.Exit1Click(Sender: TObject);
begin
  close;
end;

//-------------------------------------------------------------------------
// FormClose
//-------------------------------------------------------------------------
// Function
//    Called when Program closing, free handles if open
// Parameters
//    Sender - manadory (unused)
// Returns
//    None
// Coded
//    helen
//-------------------------------------------------------------------------
procedure TFormTerm.FormClose(Sender: TObject; var Action: TCloseAction);
begin
   // if we can't close connection then don't exit
   If Not AreDisConnected() Then  Action := caNone
end;


//-------------------------------------------------------------------------
// OPEN PORT / CLOSE PORT
//-------------------------------------------------------------------------

//-------------------------------------------------------------------------
// InitComms
//-------------------------------------------------------------------------
// Function
//     Initialise coms, reset bytes rxed/rxed and overlapped structures
// Parameters
//     None
// Returns
//    None
// Coded
//    helen
//-------------------------------------------------------------------------
procedure TFormTerm.InitComms;
begin
    BytesTxed := 0;
    BytesRxed := 0;

    // Flags for overlapped state
    fTxing := false;
    fWaitingRx := false;

    // position in incoming text buffer
    p := 1;
    Memo1.clear; 

end;

//-------------------------------------------------------------------------
// FreeComms
//-------------------------------------------------------------------------
// Function
//     Release and reset handles
// Parameters
//     None
// Returns
//    None
// Coded
//    helen
//-------------------------------------------------------------------------
procedure TFormTerm.FreeComms;
var
   DevName:string;
begin

   if hComm > INVALID_HANDLE_VALUE Then
   begin
          // release coms port
          DevName := '/dev/'+ttyn;
          ReleasePort(hComm, pchar(DevName), lockfile);
          hComm := INVALID_HANDLE_VALUE;
   end;
   // if we are disconnected, tell the user about it
   if hComm = INVALID_HANDLE_VALUE Then  SetUI(itDisConnected);

end;

//-------------------------------------------------------------------------
// SetUI
//-------------------------------------------------------------------------
// Function
//     Sets User Interface depending on state connected or not connected
// Parameters
//     State : TUIState
// Returns
//    None
// Coded
//    helen
//-------------------------------------------------------------------------
procedure TFormTerm.SetUI(State : TUIState);
begin
     // sort buttons on UI out
     case State of
     itConnected :
     begin
          DisConnect1.Enabled := true;
          TBDisconnect.Enabled := true;
          Connect1.Enabled := False;
          TBConnect.Enabled := False;
          Settings2.Enabled := true;
          TBsettings.Enabled := true;
          Amplicon1.Enabled := false;
     end;

     // we have a connection but now its gone
     itDisConnected :
     begin
          DisConnect1.Enabled := False;
          TBDisconnect.Enabled := False;
          Connect1.Enabled := True;
          TBConnect.Enabled := True;
          Settings2.Enabled := true;
          TBsettings.Enabled := True;
          Amplicon1.Enabled := True;
     end;                            

     // we have never had a connection, so we can't change properties
     itNotConnected :
     begin
          DisConnect1.Enabled := False;
          TBDisconnect.Enabled := False;
          Connect1.Enabled := False;
          TBConnect.Enabled := False;
          Settings2.Enabled := False;
          TBsettings.Enabled := False;
          Amplicon1.Enabled := True;
     end;                         

     end; // case
end;

//-------------------------------------------------------------------------
// Connect
//-------------------------------------------------------------------------
// Function
//     Open the port and set up the port properties
// Parameters
//    newttyn : Name of port
// Returns
//    None
// Coded
//    helen
//-------------------------------------------------------------------------
function TFormTerm.Connect(newttyn:string; UserComParms:TUserComParms; var error : string):Boolean;
var
  devname:string;
begin
   // If we already have a comm port open close it
   if hComm > INVALID_HANDLE_VALUE then
   begin
      Timer1.Enabled := False;
      freeComms();
   end;

   // record name
   ttyn := newttyn; 
   
   // make a device name, as we need a device name to open com9 and above
   devname := '/dev/'+newttyn;

   // Open the comm port;
   hComm := INVALID_HANDLE_VALUE;
   if  GetPrivelegeToPort(pchar(DevName), lockfile, error) Then
   begin
      hComm := FileOpen(pchar(devName), fmOpenReadWrite );
      // default error string
      if hComm <= INVALID_HANDLE_VALUE Then error := 'Failed to open port';
   end;
      
   // If we have a handle,
   if hComm > INVALID_HANDLE_VALUE Then
   begin
        // set up com port properties, previously sorted out by a call
        // to the property page dialogue
        error := '';
        SetComProperties( hComm, UserComParms);

        // initialise overlapped structures, byte counters etc
        InitComms();
        PanelRxStatus.caption := 'Bytes Rxed : 0';
        PanelTxStatus.caption := 'Bytes Txed : 0';
        PanelStatus.caption := 'Connected to ' + ttyn + ',' + inttostr(UserComParms.baud);

        // ensure we will be able to transmit data
        Memo2.Lines.BeginUpdate();
        memo2.Clear();
        Memo2.Lines.EndUpdate();

        // switch on reciever
        Timer1.Enabled := true;
        InTimer := false;
        InPaste := false;
        extra := '';
        Result := true;
    end
    else
    begin
        // not connected, reset captions
        PanelRxStatus.caption := '';
        PanelTxStatus.caption := '';
        PanelStatus.caption := 'Not Connected';
        Result := False;
    end;

end;

//-------------------------------------------------------------------------
// IfConnecedDisConnect
//-------------------------------------------------------------------------
// Function
//     If we are engaged in a conversation, disconnect us
// Parameters
//    None
// Returns
//    None
// Coded
//    helen
//-------------------------------------------------------------------------
function TFormTerm.AreDisConnected:Boolean;
begin
   If hComm > INVALID_HANDLE_VALUE Then
   begin
        If MessageDlg('Disconnect from ' + ttyn, mtInformation, [mbYes, mbNo], 0) = mrYes Then
        begin
           Timer1.Enabled := False;
           FreeComms();
           Result := True;
        end
        else
           Result := False;
   end
   else
       Result := True;
end;

//-------------------------------------------------------------------------
// NewConnection1Click
//-------------------------------------------------------------------------
// Function
//     some one selected new connection from menu
//     validate com port
//     set comm properties
//     connect to com port
// Parameters
//     Sender - mandatory (unused)
// Returns
//    None
// Coded
//    helen
//-------------------------------------------------------------------------
procedure TFormTerm.NewConnection1Click(Sender: TObject);
var
   Connected : Boolean;
   newttyn:string;
   ret : TModalResult;
   error:string;
begin
   // if we can disconnect
   If AreDisConnected() Then
   begin
       Connected := False;

       // show connection dialogue
       with TFormConnect.Create(Application) as TFormConnect do
       begin
          try
          ret := ShowModal;
          newttyn := Combobox1.Text;
          finally
          free;
          end;
       end;

       // if user said ok
       if ret = MrOk Then
       begin
           // set up port properties, by calling proprty page dialogue
           if GetWantedComProperties(newttyn, UserComParms) then
           begin
               // open the port
               if Connect(newttyn, usercomParms, error) Then  
                  Connected := True
               else
                  MessageDlg(MsgFailCon + newttyn + #13 + #10 + error, mtInformation, [mbOK], 0);
           end;
       end;

       // set up user buttons if sucessfull
       // or raise dialog if port not avaliable
       if Connected Then
       begin
          SetUI(itConnected);
       end
       else
       begin
          SetUI(itNotConnected);
       end;
   end;
end;

//-------------------------------------------------------------------------
// TBNewConnectionClick
//-------------------------------------------------------------------------
// Function
//     some one selected new connection from tool bar
//     Can chose from drop down or call a dialogue
//     connect to com port
// Parameters
//     Sender - mandatory (unused)
// Returns
//    None
// Coded
//    helen
//-------------------------------------------------------------------------
procedure TFormTerm.TBNewConnectionClick(Sender: TObject);
var
   newttyn:string;
   error:string;
begin
   // Work out com port name
   // and if selection a valid name
   //
   With Sender as TMenuItem Do
   begin
       // If we have a valid connection
       //
       case tag of
       // None selected - offer to dump current connection
       0: AreDisconnected();

       // Coms 1 to 8 are offered as menu items
       1,2,3,4,5,6,7,8:
       begin
           // get new comm port name, ttyA? 
           newttyn := ttybase + inttostr(tag-1);
           // dump existing connection if active
           If AreDisconnected() Then
           begin
              // set up port properties, by calling proprty page dialogue
              if GetWantedComProperties(newttyn, userComParms) then
              begin
                  // open port and set up buttons etc
                  if Connect(newttyn, UserComParms, error) Then
                        SetUI(itConnected)
                  else
                  begin
                        MessageDlg(MsgFailCon + ttyn + #13 + #10 + error, mtInformation, [mbOK], 0);
                        SetUI(itNotConnected);
                  end;
              end;
           end;
       end
       //
       // Other.. selected so
       // Call NewConnection dialogue
       else
          NewConnection1Click(Sender);
   end;
   end;
end;

//-------------------------------------------------------------------------
// Connect
//-------------------------------------------------------------------------
// Function
//     Connect to current com port
// Parameters
//     None
// Returns
//    None
// Coded
//    helen                          
//-------------------------------------------------------------------------
procedure TFormTerm.Connect1Click(Sender: TObject);
var
  error : string;
begin
    if Connect(ttyn, UserComParms, error) Then
          SetUI(itConnected)
    else
    begin
          MessageDlg(MsgFailCon + ttyn + #13 + #10 + error, mtInformation, [mbOK], 0);
          SetUI(itNotConnected);
    end;
end;


//-------------------------------------------------------------------------
// DisConnect
//-------------------------------------------------------------------------
// Function
//     Close the port and set up the port properties
// Parameters
//    ttyn : Name of port
// Returns
//    None
// Coded
//    helen                            
//-------------------------------------------------------------------------
procedure TFormTerm.Disconnect1Click(Sender: TObject);
begin
     Timer1.Enabled := False;
     FreeComms();
end;


//-------------------------------------------------------------------------
// SETUP
//-------------------------------------------------------------------------

//-------------------------------------------------------------------------
// Amplicon1Click
//-------------------------------------------------------------------------
// Function
//   Swap between standard TTY names and amplicon tty names
//
// Parameters
//   sender (unused)
// Returns
//    None
// Coded
//    helen
//-------------------------------------------------------------------------
procedure TFormTerm.Amplicon1Click(Sender: TObject);
var 
   i:integer;
begin
   Amplicon1.Checked :=  not Amplicon1.Checked;

   if Amplicon1.Checked Then
   begin
      ttybase := 'ttyA';
      UserComParms.fAmplicon := true;
   end
   else
   begin
      ttybase := 'tty';
      UserComParms.fAmplicon := false;
   end;

   // change pop up menu options
   for i := 0 to 7 do
   begin
      PopupComPorts.Items[i+1].Caption := ttybase + inttostr(i);
   end;   
   
end;


//-------------------------------------------------------------------------
// COMSSETUP
//-------------------------------------------------------------------------

//-------------------------------------------------------------------------
// Settings2
//-------------------------------------------------------------------------
// Function
//   Bring up dialogue to get coms settings
//
// Parameters
//   structure to hold settings
// Returns
//    None
// Coded
//    helen
//-------------------------------------------------------------------------
function TFormTerm.GetWantedComProperties(newttyn:string; var UserComParms:TUserComParms):boolean;
var
  DevName:String;
begin

   // get ports current settings
   // and determine if we can set half duplex mode
   // we can only do this if we are an Amplicon Port and root
   DevName := '/dev/'+newttyn;
   GetComProperties(DevName, UserComParms);
   // show settings dialogue
   with TFormSettings.Create(Application) do
   begin
     try
     LBPort.Caption := newttyn;     
     if ShowModal = mrOk then
     begin
        // record user comms settings 
        UserComParms.baud := strtoInt(CBBaud.Text);
        UserComParms.bits := strtoint(CBBits.Text);
        UserComParms.stopbits := strtoint(CBStopbits.Text);
        UserComParms.parity := CBParity.text;

        // sort out half duplex
        if Not RBToggle.Enabled Then
           UserComParms.halfDup := aNotSupported
        else if RBToggle.Checked Then    
           UserComParms.halfDup := aFULLDUPLEX
        else
           UserComParms.halfDup := aHALFDUPLEX;
                                              
        UserComParms.fhwflow := RBHW.Checked;
        UserComParms.fswflow := CBXOn.Checked;
        Result := true;
     end
     else
        Result := false;
     finally
     free;
     end;
  end;   
end;


//-------------------------------------------------------------------------
// Settings2
//-------------------------------------------------------------------------
// Function
//   Change com settings
//
// Parameters
//    Sender - mandatory
//    Kye    - Key pressed
// Returns
//    None
// Coded
//    helen
//-------------------------------------------------------------------------
procedure TFormTerm.Settings2Click(Sender: TObject);
var
  error:string;
begin
    // if we are connected then
   if hComm > INVALID_HANDLE_VALUE Then
   begin
        // change settings, disconnect and reconnect to kick coms
        FreeComms();
        GetWantedComProperties(ttyn, UserComParms);
        if Connect(ttyn, UserComParms, error) Then
              SetUI(itConnected)
        else
        begin
             MessageDlg(MsgFailCon + ttyn + #13 + #10 + error, mtInformation, [mbOK], 0);
             SetUI(itNotConnected);
        end;
   end
   else
   begin
        // otherwise just change properties 
        GetWantedComProperties(ttyn, UserComParms);
   end
end;

//-------------------------------------------------------------------------
// TRANSMITTING / RECIEVING
//-------------------------------------------------------------------------

//-------------------------------------------------------------------------
// FormKeyPress
//-------------------------------------------------------------------------
// Function
//   If conncted to com port, write this char to it
//
// Parameters
//    Sender - mandatory
//    Key    - Key pressed
// Returns
//    None
// Coded
//    helen
//-------------------------------------------------------------------------
procedure TFormTerm.FormKeyPress(Sender: TObject; var Key: Char);
begin
     // don't send spurious 0's 
     if key = #0 then exit;

     // if the port isn't open, forget it
     if hComm <= INVALID_HANDLE_VALUE Then exit;

     // if we are already trying to send stuff to port
     // the queue up the extra until we finish
     if InPaste or (memo2.Lines.Count > 0) then
     begin
        extra := extra + key;
        exit;
     end;
     InPaste := True;

     // add the key pressed to the end of the outgoing text
     // the timer will deal with transmitting it
     Memo2.Lines.BeginUpdate();
     memo2.lines.add(extra+key);
     Memo2.Lines.EndUpdate();
     extra := '';

     InPaste := False;
     Key := #0;
end;

//-------------------------------------------------------------------------
// Send Text File
//-------------------------------------------------------------------------
// Function
//   Dumps a text file into out going buffer
//
// Parameters
//    Sender - mandatory
// Returns
//    None
// Coded
//    helen
//-------------------------------------------------------------------------
procedure TFormTerm.SendTextFile1Click(Sender: TObject);
begin
   // find file to open
   if OpenDialog1.execute then
   begin
     SendTextFile1.Enabled := false;
     InPaste := true;
     // stuff file contents into memo2
     // timer deals with transmitting it
     memo2.Lines.beginUpdate();
     memo2.Lines.LoadFromFile(Opendialog1.FileName);
     memo2.Lines.EndUpdate();
     InPaste := false;
   end;
end;

//-------------------------------------------------------------------------
// UpdateMemo
//-------------------------------------------------------------------------
// Function
//    Interperit incoming text as simple telnet terminal
// Parameters
//    a      - text from port
//    l      - nos of characters
// Returns
//    None
// Coded
//    helen
//-------------------------------------------------------------------------
procedure TFormTerm.UpdateMemo(a : array of char; l : cardinal);
var
  c:char;
  s2:string;
  i:integer;
  sl:TstringList;
begin
   FormTerm.BytesRxed := FormTerm.BytesRxed + l;

   // fetch last line from memo box
   Memo1.Lines.BeginUpdate();
   If Memo1.Lines.Count > 0 then
      s2 := Memo1.Lines[Memo1.lines.count-1]
   else
      s2 := '';
   // create a temporary string list to work in
   SL := TStringList.Create;

   for i := 0 to l-1 do
   begin
      c := a[i];
      // If <LF> add a new line
      if c = #13 then
      begin
          SL.Add(s2);
          s2 := '';
          p := 1;
      end
      // if <CR> reset position to start of current line
      else if c = #10 then
      begin
         p := 1;
      end
      // if <BS> delete appropriate character
      else if c = #8 then
      begin
         if p > 1 then
         begin
           dec(p);
           delete(s2,p,1);
         end;
      end
      // if <DLE> delete
      else if c = #16 then
      begin
         if p > 1 then
         begin
           delete(s2,p,1);
           dec(p);
         end;
      end
      // if a valid character add it to the text
      else if c in [' '..'~', chr(9)] then
      begin
          if p < length(s2) Then
               insert(c, s2, p)
          else
               s2 := s2 + c;
          inc(p);
      end;
   end;
   SL.Add(s2);
   application.processmessages();

   // if the memo has text in it, change the last line
   If Memo1.Lines.Count > 0 then
   begin
      memo1.Lines[Memo1.Lines.Count-1] :=  SL.Strings[0];
      SL.Delete(0);
   end;

   // add remaining strings to memo
   memo1.Lines.AddStrings(SL);
   Memo1.Lines.EndUpdate();
   // throw away tempory string list
   SL.Free;

   // move memo to bottom
   Memo1.SetFocus;
   Memo1.selstart := length(memo1.Text)-1;

   // update number of bytes recieved
   PanelRxStatus.caption := 'Bytes Rxed :' + inttostr(FormTerm.BytesRxed);
   PanelRxStatus.refresh;

end;


//-------------------------------------------------------------------------
// Timer1Timer
//-------------------------------------------------------------------------
// Function
//     unload recieved chars into display
//     pump data into txmitter
// Parameters
//    None
// Returns
//    None
// Coded
//    helen
//-------------------------------------------------------------------------
procedure TFormTerm.Timer1Timer(Sender: TObject);
var
    BytesRead:integer;
    s:string;
    BytesWritten:integer;
    l :integer;
begin

    if hComm <= INVALID_HANDLE_VALUE Then exit;

    // check re-entrancy
    if InTimer Then exit;
    InTimer := true;

    Timer1.Enabled := False;

    // deal with recieve data

    // if we have incoming coms data update the display
    if DataAvailable(hComm,0) Then
    begin
        BytesRead := FileRead( hComm, RxBuffer, 1024);
        if BytesRead > 0 Then UpdateMemo(Rxbuffer, BytesRead);
    end;

    // deal with transmit data
    // if theres data to send and we are not in flux
    if (Memo2.lines.Count > 0) and (not InPaste) then
    begin
         // get the next line of info to go out of the buffer
         // if theres only one line left don't send a CR LF
         InPaste := true;
         Memo2.Lines.BeginUpdate();
         if memo2.lines.count > 1 Then
            s :=  Memo2.Lines[0] + #10 + #13
         else
            s :=  Memo2.Lines[0];

         // copy the line (or 1024 bytes) out to the com port
         strplcopy(txbuffer, s, 1024);

         if length(s) <= 1024 then
            l := length(s)
         else
            l := 1024;

         // send the line over the coms port
         BytesWritten := FileWrite(hComm, txbuffer, l);

         // remove the line we've done with it
         Memo2.Lines.Delete(0);
         Memo2.Lines.EndUpdate();
         InPaste := false;

         // bytes transmitted and leave the ftxing falg clear
         if BytesWritten > 0 Then
         begin
             BytesTxed := BytesTxed  + bytesWritten;
             PanelTxStatus.caption := 'Bytes Txed : ' + inttostr(BytesTxed);
         end;
    end;

    // If the outgoing memo box is empty, allow the user to send a file
    if (Memo2.Lines.Count = 0) and not fTxing Then
    begin
        SendTextFile1.Enabled := true;
    end;

    // clear the re-entrancy flag
    InTimer := False;
    Timer1.Enabled := true;

end;


//-------------------------------------------------------------------------
// Editing
//-------------------------------------------------------------------------

//-------------------------------------------------------------------------
// Copy1Click
//-------------------------------------------------------------------------
// Function
//    Copy selection in memo box to clip board
// Parameters
//    Sender (mandatory)
// Returns
//    None
// Coded
//    helen
//-------------------------------------------------------------------------
procedure TFormTerm.Copy1Click(Sender: TObject);
begin
   try
   if (memo1.lines.count > 0) and (Memo1.SelLength > 0) then  memo1.CopyToClipBoard;
   finally
   end;
end;

//-------------------------------------------------------------------------
// SellectAll1Click
//-------------------------------------------------------------------------
// Function
//    Select all text in memo box
// Parameters
//    Sender (mandatory)
// Returns
//    None
// Coded
//    helen
//-------------------------------------------------------------------------
procedure TFormTerm.SelectAll1Click(Sender: TObject);
begin
     try
     Memo1.SelectAll;
     finally
     end;
end;

//-------------------------------------------------------------------------
// Paste1Click
//-------------------------------------------------------------------------
// Function
//    paste clipboard to memo2 so it gets sent
// Parameters
//    Sender (mandatory)
// Returns
//    None
// Coded
//    helen
//-------------------------------------------------------------------------
procedure TFormTerm.Paste1Click(Sender: TObject);
begin
   // paste the clipboard into memo2 and the timer transmits it
   if InPaste Then exit;
   InPaste := True;
   Memo2.Lines.BeginUpdate();
   Memo2.PastefromClipBoard;
   Memo2.Lines.EndUpdate();
   InPaste := false;
end;

//-------------------------------------------------------------------------
// ClearBuffer1Click
//-------------------------------------------------------------------------
// Function
//    Empty Memo1
// Parameters
//    Sender (mandatory)
// Returns
//    None
// Coded
//    helen
//-------------------------------------------------------------------------
procedure TFormTerm.ClearBuffer1Click(Sender: TObject);
begin
  Memo1.Lines.Clear;
  p := 1;
end;


//-------------------------------------------------------------------------
// ViewTextFile1Click
//-------------------------------------------------------------------------
// Function
//    load a file into memo1 so user can select bits and from it and
//    send its etc etc
// Parameters
//    Sender (mandatory)
// Returns
//    None
// Coded
//    helen
//-------------------------------------------------------------------------
procedure TFormTerm.ViewTextFile1Click(Sender: TObject);
begin
   if OpenDialog1.execute then
   begin
     try
     memo1.Lines.LoadFromFile(Opendialog1.FileName);
     except
     end;
     p := 1;

   end;
end;

//-------------------------------------------------------------------------
// Print
//-------------------------------------------------------------------------
procedure TFormTerm.Print1Click(Sender: TObject);
var
  r: TRect;
  i: Integer;
begin
    with Printer do
    begin
      r := Rect(200,200,(Pagewidth - 200),(PageHeight - 200));
      BeginDoc;
      for i := 0 to Memo1.Lines.Count do
      begin
          Canvas.TextOut( 200 
                        , 200 + (i * Canvas.TextHeight(Memo1.Lines.Strings[i]))
                        , Memo1.Lines.Strings[i]
                        );
      end;
      EndDoc;    
    end;
end;

//-------------------------------------------------------------------------
// Save to file
//-------------------------------------------------------------------------

//-------------------------------------------------------------------------
// S1Click
//-------------------------------------------------------------------------
// Function
//    Save contents of memo1 to a file
// Parameters
//    Sender (mandatory)
// Returns
//    None
// Coded
//    helen
//-------------------------------------------------------------------------
procedure TFormTerm.S1Click(Sender: TObject);
begin
     If SaveDialog1.Execute Then
     begin
          Memo1.Lines.SaveToFile( SaveDialog1.FileName);
     end;
end;

//-------------------------------------------------------------------------
// Formating
//-------------------------------------------------------------------------

//-------------------------------------------------------------------------
// Font1Click
//-------------------------------------------------------------------------
// Function
//    Allow user to muck about with font
// Parameters
//    Sender (mandatory)
// Returns
//    None
// Coded
//    helen
//-------------------------------------------------------------------------
procedure TFormTerm.Font1Click(Sender: TObject);
begin
    if FontDialog1.execute() Then
       Memo1.Font := FontDialog1.Font;
end;

//-------------------------------------------------------------------------
// Help
//-------------------------------------------------------------------------

//-------------------------------------------------------------------------
// AboutMEVTerminal1Click
//-------------------------------------------------------------------------
// Function
//    Show about box
// Parameters
//    Sender (mandatory)
// Returns
//    None
// Coded
//    helen
//-------------------------------------------------------------------------
procedure TFormTerm.AboutMEVTerminal1Click(Sender: TObject);
begin
     with TFormAbout.Create(application) do
     begin
        try
        ShowModal();
        finally
        free;
        end;
    end;
end;

end.

