//
// MEV Terminal Program
//
// Version 1.03
//
// PROJECT DESCRIPTION
//
// Kylix Comms Example using Linux Comms I/F
//
// This source code is supplied as an example of a aerial communications
// in Kylix for the Amplicon range of serial cards.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version
// 2 of the License, or (at your option) any later version.
// Although the best endevours have been made to ensure the accuracy
// of this code, MEV make no warranty of any kind with regard to this
// sample source code.
//
// by MEV Ltd
// Copyright (c) 1997-2000 MEV Limited
// Suite 4 Baxall Business Centre, Unit 25b,
// Adswood Road Ind Est, STOCKPORT, Cheshire. SK3 8LF
// +44 (0)161 477 1898
// @mev-ltd.demon.co.uk
//
// ENVIRONMENT
//    Compiler:      Borland KyLix Professional Version 1.0
//    Target system: PC, running Linux
//
// AMENDMENT RECORD
//
// started    1.01  27/09/2001
//
// KNOWN LIMITATIONS / PROBLEMS
//
// Can't paste more that 64K into out going buffer
// A single line can't be more than 1024K
//
// Module contents
//
// Low level Linux OS code for talking to serial ports
// plus code to get privilege to open the ports in the first place
//
// NB to run this code the executeable must belong to the uucp group and
// have system priveleges
//
// chgrp uucp MEVTerminal
// chmod g+s MEVTerminal
//

unit UnitSerial;

interface

uses
  SysUtils, Types, Classes, Variants, libc, kernelioctl;

type  
  //
  // User communication settings
  //
  THalfDupModes = (aNotSupported, aNone, aFULLDUPLEX,aHALFDUPLEX,aSLAVEMULTIPLEX);
  
  TUserComParms = record
  fAmplicon:boolean;
  baud :integer;
  bits : integer;
  parity:string;
  stopbits:integer;
  halfDup:THalfDupModes;
  fhwflow:boolean;
  fswflow:boolean;
  end;
  
// serial communications functions
function DataAvailable(hPort, tmout:integer):boolean;
procedure SetComProperties(hPort:integer; var  UserComParms:TUserComParms);
procedure GetComProperties(devname:string; var  UserComParms:TUserComParms);
function GetSupportedBauds( var bauds :array of integer):integer;

// system functions to secure privilege to port
function  InitialiseProcessIDs : boolean;
function   GetPrivelegeToPort(ttyn:pchar; var lockfile, error :string):boolean;
procedure  ReleasePort(portfd :integer; ttyn:pchar; lockfile :string);
  
const
  INVALID_HANDLE_VALUE = -1;
{$if CS7 <>  $0000020 }
  rCS7       = $0000020;      // CS7 is defined incorrectly in LibC.pas
{$else}
  rCS7       = CS7;
{$ifend}  
  
var
   Lockfile_location : string = '/var/lock';

implementation

const
  //
  // Non-standard IOCTLs for RS485 line modes in the ampser driver.
  // Connect Tech also use these values to support their Blue Heat cards
  // (see <ftp://ftp.connecttech.com/pub/linux/blueheat/>).
  //
  TIOCSER485SET =	$054A0;	// Set the 485 line mode 
  TIOCSER485GET =	$054A1;	// Get the 485 line mode 

  TIOCSER485NOT_INITED	   =	0;
  TIOCSER485FULLDUPLEX	   =	1;
  TIOCSER485HALFDUPLEX	   =	2;
  TIOCSER485SLAVEMULTIPLEX =	3;

  //
  // Non-standard IOCTLs for the Amplicon serial driver.
  //

  // Flags set and get. 
  TIOCAMPSERFLAGSET       =	$04100;	// Set flags 
  TIOCAMPSERFLAGGET	  =     $04101;	// Get flags 

  // file mode settings
  mode_r = 4;
  mode_w = 2;
  mode_x = 1;
  mode_rw = (mode_r or mode_w);
  mode_rwx = (mode_r or mode_w or mode_x);
  grp_user = 6;
  grp_grp = 3;
  grp_others = 0;

var
  real_uid: uid_t;
  real_gid: gid_t;
  port_uid: uid_t;
  port_gid: gid_t;
  eff_uid: uid_t;
  eff_gid: gid_t;
  username: string;
  homedir : string;
  not_suid:integer = -1;
  
//-------------------------------------------------------------------------
// SERIAL COMMS
//-------------------------------------------------------------------------

//-------------------------------------------------------------------------
// DataAvailable
//-------------------------------------------------------------------------
// Function
//     Check if there is data pending.
// Parameters
//     Port
//     time out in uS
// Returns
//     1 if data pending
// Coded
//     helen
//-------------------------------------------------------------------------
function DataAvailable(hPort, tmout:integer):boolean;
var
  tv : TTimeVal;
  fds: Tfdset;

begin
  tv.tv_sec := Trunc(tmout / 1000);
  tv.tv_usec := Trunc(tmout mod 1000) * 1000;

  FD_ZERO(fds);
  FD_SET(hPort, fds);

  Result := False;
  if (select(hPort+1, @fds, nil, nil, @tv) > 0) Then
  begin
        if FD_ISSET(hPort, fds) Then Result := True;
  end;
end;

//-------------------------------------------------------------------------
// SetHWF
//-------------------------------------------------------------------------
// Function
//     Set hardware flow control. Needs to be done after RTS raised
// Parameters
//     Port
//     hwf  - hard ware flow control
// Returns
//
// Coded
//     helen
//-------------------------------------------------------------------------
procedure sethwf(hPort : integer; hwf:boolean);
var
  tty : termios;
begin

  tcgetattr(hport, tty);
  if hwf Then
	tty.c_cflag := tty.c_cflag or CRTSCTS
  else
	tty.c_cflag :=  tty.c_cflag and not CRTSCTS;

  tcsetattr(hPort, TCSANOW, tty);

end;

//-------------------------------------------------------------------------
// SetComProperties
//-------------------------------------------------------------------------
// Function
//     Sets com port to users property settings
//     etc
// Parameters
//     Port  - set to invalid if not available yet
//     baud  - baud rate
//     bits  - number of bits
//     stopbits - number of stop bits
//     parity - 'E' = even, 'O' = odd, 'N' = none, 'M' = mark and 'S' = space
//     sfw    - software flow control
//     tty
//
// Returns
//     nothing
// Coded
//     helen
//-------------------------------------------------------------------------
procedure SetComProperties(hPort:integer; var  UserComParms:TUserComParms);
//baud, bits:integer; stopbits:integer;parity:string; hwf, swf:boolean);
var
  tty       : Termios;
  Par       : string;
  spd       : speed_t;
  duplex    : integer;
begin

  if hPort <> INVALID_HANDLE_VALUE then tcgetattr(hport, tty);

  par := uppercase(copy(UserComParms.parity,1,1)) ;

  // We generate mark and space parity ourself.
  if (UserComParms.bits = 7) and ((Par = 'M') or (Par = 'S')) Then UserComParms.bits := 8;

  case UserComParms.baud div 100 of
    0:    spd := B0;
    3:    spd := B300;
    6:    spd := B600;
    12:	 spd := B1200;
    24:	 spd := B2400;
    48:	 spd := B4800;
    96:	 spd := B9600;
    192:  spd := B19200;
    384:  spd := B38400;
    576:  spd := B57600;
   1152:  spd := B115200;
   2304:  spd := B230400;
   4608:  spd := B460800;
   5000:  spd := B500000;
   5760:  spd := B576000;
   9216:  spd := B921600;
   10000: spd := B1000000;
   11520: spd := B1152000;
   15000: spd := B1500000;
   20000: spd := B2000000;
   25000: spd := B2500000;
   30000: spd := B3000000;
   35000: spd := B3500000;
   40000: spd := B4000000;
  else
  begin
        spd := B9600;
        // couldn't do baud rate
        UserComParms.baud := 9600;
  end;
  end; // case

  cfsetospeed(tty, spd);
  cfsetispeed(tty, spd);

  case UserComParms.bits of
  5: tty.c_cflag := ((tty.c_cflag and ( not CSIZE))) or CS5;
  6: tty.c_cflag := ((tty.c_cflag and ( not CSIZE))) or CS6;
  7: tty.c_cflag := ((tty.c_cflag and ( not CSIZE))) or rCS7;
  8: tty.c_cflag := ((tty.c_cflag and ( not CSIZE))) or CS8;
  end;

  // Set into raw, no echo mode
  tty.c_iflag :=  IGNBRK;
  tty.c_lflag := 0;
  tty.c_oflag := 0;
  tty.c_cflag := tty.c_cflag  or (CLOCAL or CREAD);
  tty.c_cflag := tty.c_cflag and not CRTSCTS;
  tty.c_cc[VMIN] := chr(1);
  tty.c_cc[VTIME] := chr(5);

  // software flow control
  if UserComParms.fswflow then
        tty.c_iflag := tty.c_iflag or (IXON or IXOFF)
  else
	tty.c_iflag := tty.c_iflag and (not (IXON or IXOFF or IXANY));

  tty.c_cflag := tty.c_cflag and (not (PARENB or PARODD));

  // Parity
  if par = 'E' Then
	tty.c_cflag := tty.c_cflag or PARENB
  else if par = 'O' then
	tty.c_cflag := tty.c_cflag or (PARENB or PARODD);

  // Stop bits
  if UserComParms.stopbits = 2 then
    tty.c_cflag :=  tty.c_cflag  or CSTOPB
  else
    tty.c_cflag :=  tty.c_cflag  and not CSTOPB;


  if hPort <> INVALID_HANDLE_VALUE then
  begin
    tcsetattr(hPort, TCSANOW, tty);
    sethwf(hPort, UserComParms.fhwflow);
  end;

  // set half duplex mode
  if UserComParms.fAmplicon Then
  begin
    case UserComParms.halfDup of
    aNONE :           duplex := TIOCSER485NOT_INITED;
    aFULLDUPLEX :     duplex := TIOCSER485FULLDUPLEX;
    aHALFDUPLEX :     duplex := TIOCSER485HALFDUPLEX;
    aSLAVEMULTIPLEX : duplex := TIOCSER485SLAVEMULTIPLEX;
    else
        duplex := -1;
    end;

    if duplex >=0 then
    begin
        if ioctl(hPort,  TIOCSER485GET, @duplex ) <> 0 Then UserComParms.halfDup := aNotSupported;
    end;    
  
  end;
  
end;


//-------------------------------------------------------------------------
// GetComProperties
//-------------------------------------------------------------------------
// Function
//     Set up termios structure holding baudrate, parity and number of bits
//     etc
// Parameters
//     Port  - set to invalid if not available yet
//     baud  - baud rate
//     bits  - number of bits
//     stopbits - number of stop bits
//     parity - 'E' = even, 'O' = odd, 'N' = none, 'M' = mark and 'S' = space
//     sfw    - software flow control
//     tty
//
//
// Returns
//     nothing
// Coded
//     helen
//-------------------------------------------------------------------------
procedure GetComProperties(devname:string; var  UserComParms:TUserComParms);
var
  tty       : Termios;
  hPort     : integer;
  tlockfile  : string;
  error     : string;
  duplex    : integer;
begin
   if  GetPrivelegeToPort(pchar(devname), tlockfile, error) Then
   begin
//     hPort := Open(pchar(devName), O_RDWR or O_NONBLOCK );
     hPort := FileOpen(pchar(devName), fmOpenRead );
     
     if hPort > INVALID_HANDLE_VALUE then 
     begin
        tcgetattr(hport, tty);

        case cfgetispeed(tty) of
          B0:       UserComParms.baud := 0;
          B300:     UserComParms.baud := 300;
          B600:     UserComParms.baud := 600;
          B1200:    UserComParms.baud := 1200;
          B2400:    UserComParms.baud := 2400;
          B4800:    UserComParms.baud := 4800;
          B9600:    UserComParms.baud := 9600;  
          B19200:   UserComParms.baud := 19200; 
          B38400:   UserComParms.baud := 38400; 
          B57600:   UserComParms.baud := 57600; 
          B115200:  UserComParms.baud := 115200;
          B230400:  UserComParms.baud := 230400;
          B460800:  UserComParms.baud := 460800;
          B500000:  UserComParms.baud := 500000;
          B576000:  UserComParms.baud := 576000;
          B921600:  UserComParms.baud := 921600;
          B1000000: UserComParms.baud := 1000000;
          B1152000: UserComParms.baud := 1152000;
          B1500000: UserComParms.baud := 1500000;
          B2000000: UserComParms.baud := 2000000;
          B2500000: UserComParms.baud := 2500000;
          B3000000: UserComParms.baud := 3000000;
          B3500000: UserComParms.baud := 3500000;
          B4000000: UserComParms.baud := 4000000;
        else
          UserComParms.baud := 9600;
        end;

        case (tty.c_cflag and CSIZE) of
        CS5: UserComParms.bits := 5;
        CS6: UserComParms.bits := 6;
        rCS7: UserComParms.bits := 7;
        CS8: UserComParms.bits := 8;
        end;

        if (tty.c_iflag and (IXON or IXOFF) ) <> 0 then
            UserComParms.fswflow := true
        else
            UserComParms.fswflow := false;

        case tty.c_cflag and (PARENB or PARODD) of
        PARENB  : UserComParms.parity := 'Even';
        PARENB or PARODD: UserComParms.parity := 'Odd';
        else
          UserComParms.parity := 'None';
        end;

        // Stop bits
        if (tty.c_cflag and CSTOPB) <> 0 then
           UserComParms.stopbits := 2
        else
           UserComParms.stopbits := 1;

        // half duplex mode
        if UserComParms.fAmplicon Then
        begin
         duplex := -1;
      	 if ioctl(hPort, TIOCSER485GET, @duplex) = 0 then
          begin
            // fish out duplex setting
            case Duplex of
            TIOCSER485NOT_INITED	    :	UserComParms.halfDup := aNone;
            TIOCSER485FULLDUPLEX	    :	UserComParms.halfDup := aFULLDUPLEX;
            TIOCSER485HALFDUPLEX	    :	UserComParms.halfDup := aHALFDUPLEX;
            TIOCSER485SLAVEMULTIPLEX :	UserComParms.halfDup := aSLAVEMULTIPLEX;
            else
               UserComParms.halfDup := aNotSupported;
            end;    
            
          end
          else
             UserComParms.halfDup := aNotSupported;
        
        end;
     end;
     ReleasePort(hport, pchar(devname), tlockfile);
  end;


end;


//-------------------------------------------------------------------------
// GetSupportedBauds
//-------------------------------------------------------------------------
// Function
//     determines supported baud rate
// Parameters
//     bauds - array to hold baud rates
// Returns
//     nothing
// Coded
//     helen
//-------------------------------------------------------------------------
function GetSupportedBauds( var bauds :array of integer):integer;
var
  n : integer;
begin

  // zero array to start
  for n := 0 to high(bauds) do bauds[n] := 0;

  // standard bauds, supported by all cards
  //
  n := 6;
  if high(bauds) >= 0 then  bauds[0] := 300;
  if high(bauds) >= 1 then  bauds[1] := 600;
  if high(bauds) >= 2 then  bauds[2] := 1200;
  if high(bauds) >= 3 then  bauds[3] := 2400;
  if high(bauds) >= 4 then  bauds[4] := 4800;
  if high(bauds) >= 5 then  bauds[5] := 9600;
  
  // comment out if baud particular rate not to be supported
  // -----------------------------------------
  if high(bauds) >= n then bauds[n] := 19200; 
  inc(n);
  // -----------------------------------------
  if high(bauds) >= n then bauds[n] := 38400;
  inc(n);
  if high(bauds) >= n then bauds[n] := 57600;
  inc(n);
  if high(bauds) >= n then bauds[n] := 115200;
  inc(n);
  if high(bauds) >= n then bauds[n] := 230400;
  inc(n);
  if high(bauds) >= n then bauds[n] := 460800;
  inc(n);
  if high(bauds) >= n then bauds[n] := 500000;
  inc(n);
  if high(bauds) >= n then bauds[n] := 576000;
  inc(n);
  if high(bauds) >= n then bauds[n] := 921600;
  inc(n);
  if high(bauds) >= n then bauds[n] := 1000000;
  inc(n);
  if high(bauds) >= n then bauds[n] := 1152000;
  inc(n);
  if high(bauds) >= n then bauds[n] := 1500000;
  inc(n);
  if high(bauds) >= n then bauds[n] := 2000000;
  inc(n);
  if high(bauds) >= n then bauds[n] := 2500000;
  inc(n);
  if high(bauds) >= n then bauds[n] := 3000000; 
  inc(n);
  if high(bauds) >= n then bauds[n] := 3500000;
  inc(n);
  if high(bauds) >= n then bauds[n] := 4000000;
  inc(n);
         
  result := n;
end;

//-------------------------------------------------------------------------
// PRIVELEGES
//-------------------------------------------------------------------------
                
//-------------------------------------------------------------------------
//  InitialiseProcessIDs
//-------------------------------------------------------------------------
// Function
//     Get an id for our application so that we can take over the ports
//     we need
// Parameters
//     none
// Returns
//
// Coded
//     helen
//-------------------------------------------------------------------------
function  InitialiseProcessIDs : boolean;
var
  pwd  :ppasswordrecord;		// To look up user name 
  s:string;
begin
   real_uid := getuid();
   real_gid := getgid();
   eff_uid  := geteuid();
   eff_gid  := getegid();

  // Avoid fraude 

  // Get password file information of this user. 
  pwd := getpwuid(real_uid);
  if pwd = nil then
  begin
      // user doesn't exist???
      result := false;
      exit;
  end;

  // Remember home directory and username. 
  s := getenv('HOME');
  if s = '' then
      homedir := pwd^.pw_dir
  else
      homedir := s;

  username :=pwd^.pw_name;
  
  Result := true; 
end;

//-------------------------------------------------------------------------
//  set_privs
//-------------------------------------------------------------------------
// Function
//    Set priviliges (swap uid's) 
// Parameters
//     none
// Returns
//
// Coded
//     helen
//-------------------------------------------------------------------------
procedure set_privs;
begin
  setregid(real_gid, eff_gid);
  if (setreuid(real_uid, eff_uid) < 0) then
  begin
//	fprintf(stderr, "minicom: cannot setreuid(%d, %d)\n", real_uid, eff_uid);
  end; 
  not_suid := 0;
end;

//-------------------------------------------------------------------------
//  drop_privs
//-------------------------------------------------------------------------
// Function
//     Drop priviliges (swap uid's) 
// Parameters
//     none
// Returns
//
// Coded
//     helen
//-------------------------------------------------------------------------
procedure drop_privs;
begin
  setregid(eff_gid, real_gid);
  if (setreuid(eff_uid, real_uid) < 0) Then
  begin
//	fprintf(stderr, "minicom: cannot setreuid(%d, %d)\n", eff_uid, real_uid);
  end; 
  not_suid := 1;
end;

//-------------------------------------------------------------------------
//  GetPrivelegeToPort
//-------------------------------------------------------------------------
// Function
//     Create a lock file for this port
//     bump privelege so we can open this port
//     remember ports settings so we can restore it
//
// Parameters
//     ttyn - name of port
//     lockfile - variable to recieve lock file name
// Returns
//     true if sucessful
// Coded
//     helen
//-------------------------------------------------------------------------
function   GetPrivelegeToPort(ttyn:pchar; var lockfile, error :string):boolean;
const
  rw_rw_rw_ = (mode_rw shl grp_user) or  (mode_rw shl grp_grp) or   (mode_rw shl grp_others) ; // 0666 in octal
      w__w_  = (mode_w shl grp_grp ) + (mode_w shl grp_others); // 022 in octal
var
  stt      : _stat;
  fd, n    : integer;
  e, b, i  : integer;
  pid      : pid_t;
  mask     : integer;
  buf      : array [0..127] of char;
  s        : string;
  portfd   : integer;
  locked   : boolean;
  
begin

  // Upgrade our status.
  set_privs();

  // First see if the lock file directory is present. */
  if (LockFile_Location <> '') and (stat(pChar(LockFile_location), stt) = 0) then
  begin
{$if defined(SVR4_LOCKS)}
		stat(pChar(ttyn), stt);
      lockFile := LockFile_Location + '/LK..'
                  + format('%03d',[major(stt.st_dev)]) 
                  + format('%03d',[major(stt.st_rdev)]) 
                  + format('%03d',[minor(stt.st_rdev)])
                  + #0;
      
{$else} // SVR4_LOCKS 
      lockFile := LockFile_Location + '/LCK..'+ extractFilename(ttyn) + #0;
{$ifend} // SVR4_LOCKS 
  end 
  else
  begin
   	lockfile := '';
  end;

  // if we have a lock file
  if (lockfile <> '') Then
  begin
     fd := FileOpen(pchar(lockfile), O_RDONLY);
  
     // if the file exists
     // read the contents of the lock file if it exists
     if fd >= 0 then
     begin
        n := FileRead(fd, buf, 127);
        FileClose(fd);
      
        locked := true; 
      
        // try and release it
        if (n > 0) Then
        begin
           s := strpas(buf);
           
           // sort out where PID number is in string
           e := 0;
           b := 0;
           for i := 1 to length(s) do
           begin
               if (e=0) and not (s[i] in [' ', '0'..'9']) then e := i;
               if (b=0) and not (s[i] in [' ']) Then b := i; 
           end;
           
           try
           pid := StrtoInt(copy(s,b,e-b-1));
           except
           pid := -1;
           end;

           // see if the process is stale
           if pid = getpid() Then
           begin
               locked := false;
               unlink(PChar(lockfile));
           end
           else if (pid > 0) Then
           begin
               // pid varaible  set zero if stale
               if (kill(pid, 0) < 0) then
               begin
                   if (errno = ESRCH) Then
                   begin
                       locked := false;
                       unlink(PChar(lockfile));
                   end 
               end
                   
           end;
        end;
      
        if locked Then
        begin
           error := 'Device is locked by ' + s;
           drop_privs();
           Result := false;
           exit;
        end;
     end;
  end;

  // if we successfully got a lock file
  if lockfile <> '' Then
  begin
     	// Create lockfile compatible with UUCP-1.2 
    	mask := umask(w__w_);
{$if defined(_COH3)}
       fd := creat(pChar(lockfile), rw_rw_rw_);
{$else}
       fd := open(pchar(lockfile), O_WRONLY or O_CREAT or O_EXCL, rw_rw_rw_);
{$ifend}
       if fd < 0 then
       begin
          error := 'Cannot create lockfile ' + lockfile;
          case errno of
          ENOENT : error := error + '- no such file or directory';
          EACCES : error := error + '- Permission Denied';
          EEXIST : error := error + '- file exists';
          end;
          drop_privs();
          Result := false;
          exit;
       end;
       umask(mask);
       chown(pChar(lockfile), real_uid, real_gid);

       pid := getpid();   
       s := format('%010d', [pid]) + ' mevterminal ' + format('%.20s', [username]);
       strplcopy(buf,s,127);
       n := length(s);
       if n > 127 Then
       begin
         n := 127;
         buf[127] := #0;
       end;
       Filewrite(fd, buf, n);
       Fileclose(fd);
  end;

  // Now open the tty device. 
{$if defined(O_NDELAY) && defined(F_SETFL)}
	portfd := FileOpen(ttyn, O_RDWR or O_NDELAY);
	if portfd > INVALID_HANDLE_VALUE then
   begin
		// Cancel the O_NDELAY flag. 
		n := fcntl(portfd, F_GETFL, 0);
		fcntl(portfd, F_SETFL, n and not O_NDELAY);
	end
{$else}
	portfd := FileOpen(ttyn, O_RDWR);
{$ifend}
	if (portfd > INVALID_HANDLE_VALUE ) then
   begin
//		m_savestate(portfd);
//		port_init();
   end;
{$if defined(__linux__) || (defined(BSD) && (BSD >= 199306))}
  s_errno := errno;
{$ifend}
  if (portfd <= INVALID_HANDLE_VALUE) then
  begin
      error := 'Can not open port';
		if lockfile <> '' Then unlink(pChar(lockfile));
		drop_privs();
		Result := false;
      exit;
  end;

  // Remember owner of port 
  stat(pChar(ttyn), stt);
  port_uid := stt.st_uid;
  port_gid := stt.st_gid;

  // Give it to us! 
  if (real_uid <> 0) Then chown(pchar(ttyn), real_uid,  real_gid);

  // Set CLOCAL mode 
  //m_nohang(portfd);

  // Set Hangup on Close if program crashes. (Hehe) 
  //m_hupcl(portfd, 1);
  //m_flush(portfd);
  drop_privs();
  result := True;
end;

//-------------------------------------------------------------------------
//  ReleaseLockFile
//-------------------------------------------------------------------------
// Function
//     Release lock file for this port
//
// Parameters
//     lockfile - lock file name
// Returns
//     true if sucessful
// Coded
//     helen
//-------------------------------------------------------------------------
procedure  ReleasePort(portfd :integer; ttyn:pchar; lockfile :string);
begin
   if (portfd > 0) then	
   begin
   	Fileclose(portfd);
   end;   

   set_privs();
   if lockfile <> '' then  unlink(Pchar(lockfile));
   if (real_uid <> 0) then chown(pchar(ttyn), port_uid, port_gid);
 	drop_privs();
end;



end.
