//
// Amplicon NT DIO Driver
//
// $Id$
//
// Copyright (c) 1996-2001
// by MEV Limited, Suite 8, Baxall Business Centre, 
//                 Adswood Road Industrial Estate,
//                 Stockport, SK3 8LF, UNITED KINGDOM.
// +44 161 477 1898
// www.mev.co.uk
//
// This source code is supplied as an example of TCsetUserInterrupt
// It may be freely re-used and modified, providing 
// relevant copyright is retained by MEV. Although the best endevours have 
// been made to ensure the accuracy of this code, MEV make no 
// warranty of any kind with regard to this sample source code.
//
// All rights reserved.
// Started 21 December 1997
//
// MODULE CONTENTS
//
// PPI Mode 1 example
//
// DESCRIPTION
//
// Example program that uses the PPI interrupt to poll port A of PPI
// in mode 1
//
// ENVIRONMENT 
//
// Compiler:      Microsoft Visual C++ V4.0
// Target System: PC, running MS-Windows NT v4.0
//
// AMENDMENT RECORD
//
// 2001-12-05
//
//		Added Sleep(0) calls before transmitting the first byte to allow
//		the user interrupt threads to start running first. (IJA)
//
//		Display number of data errors. (IJA)
//
//		Return a status from main(). (IJA)
//
//
//
#include <windows.h>
#include <stdio.h>
#include <conio.h>
#include <time.h>
#include "adioctl.h"
#include "dio_tc.h"

#define M0_AIN_BOUT     0x90   // 10010000
#define M1_AIN_BOUT     0xB4   // 10110100

#define ENABLE_RIEA_INTE    ((4<<1)|1) // bit 4 Port A read INTE
#define DISABLE_RIEA_INTE    ((4<<1))  // bit 4 Port A read INTE

#define ENABLE_WIEA_INTE    ((6<<1)|1) // bit 4 Port A write INTE
#define DISABLE_WIEA_INTE    ((6<<1))  // bit 4 Port A write INTE

#define ENABLE_B_INTE    ((2<<1)|1) // bit 2 Port B INTE
#define DISABLE_B_INTE    ((2<<1))  // bit 2 Port B INTE


ULONG ntxed = 0;
ULONG nrxed = 0;
ULONG nerrs = 0;

/*============================================================================
; WriteProcessEvent
;=============================================================================
;Function:
;   This routine is called from the IRQ service routine when
;   PPI X1 C0 changes state. It writes data to PPI port B
;   and strobes it
;Formal Parameters:
;   h       card handle
;   wParam  User parameter is card address in this case
;   data    Contents of PPI ABC at interrupt   
;Returns
;    None
;---------------------------------------------------------------------------
*/
void CALLBACK WriteProcessEvent(const short h, const WPARAM wParam, const ULONG PortC)
{
	// write out data on port B
	DIOsetDataEx(h, PPIX, 1, (BYTE)ntxed++);  // 70uS
	// strobe it out
	DIOsetModeEx(h, PPIX, (6<<1));	   // set bit 6 (strobe) 00001101 70uS
	DIOsetModeEx(h, PPIX, ((6<<1)|1));  // clr bit 6 (strobe) 00001101 70uS
}


/*============================================================================
; ReadProcessEvent
;=============================================================================
;Function:
;   This routine is called from the IRQ service routine when
;   PPI X1 C3 changes state. It prints the data strobed into
;   PPI port A
;Formal Parameters:
;   h       card handle
;   wParam  User parameter is card address in this case
;   data    Contents of PPI ABC at interrupt   
;Returns
;    None
;---------------------------------------------------------------------------
*/
void CALLBACK ReadProcessEvent(const short h, const WPARAM wParam, const ULONG PortC)
{
	short data;    

	if(nrxed == 0)
	{
		DIO_TC_restorenormalpriority();
	}

	// read in data on port A
	DIOgetDataEx(h, PPIX, 0, &data);  // 70uS

	if(data != (BYTE)nrxed++)
	{
		nerrs++;
	}

	// acknowledge it
	DIOsetModeEx(h, PPIX, (7<<1));	   // set bit 7 (ack) 00001110 70uS
	DIOsetModeEx(h, PPIX, ((7<<1) | 1));   // clr bit 7 (ack) 00001111 70uS
}


/*============================================================================
; printusage
;=============================================================================
;Function:
;   Print how to use program
;Formal Parameters:
;   none
;Returns
;    None
;---------------------------------------------------------------------------
*/
void printusage(void)
{
	printf("\n\r\n\rUsage = PPIIRQM1 <BoardType> <address> <irq>");
	printf("\n\r\n\rwhere:");
	printf("\n\r\tBoardType = 36,212,214,215,272");
	printf("\n\r\taddress = base address (in Hex) eg 0x300");
	printf("\n\r\tirq = irq vector");
	printf("\n\r\neg:\n\r\tppiirq 214 300 5\n\r\n\rtests a PC214E card installed at 0x300 on IRQ 5\n\r\n\r");
	printf("\n\rData is strobed from port B to Port A");
	printf("\n\rYou will need to wire PPI port A to Port B");
	printf("\n\rPC6 to PC4 (strobe) and PC7 to PC2 (ack)");
}

/*============================================================================
; main
;=============================================================================
;Function:
;   main execution loop of console app
;   for strobed data transfer
;	as strobes are generated at app level
;   and there is 250us latency associated with
;	each IRQ service we can excpect data rates
;	of around 2 kBytes per second 
;
;Formal Parameters:
;   argc    number of arguments past
;   argv[]  array of strings containing arguments
;Returns
;    None
;---------------------------------------------------------------------------
*/
int main(int argc, LPSTR argv[])
{
	short	hBoard;
	int cardtype, address, irq;
	short hTCINTPortA;
	short hTCINTPortB;
	BOOL going;
	clock_t start, finish;
	double duration;

	printf("\n\rPPIIRQ is a quick console application to demonstrate");
	printf("\n\rreading and writing to PPI in Mode 1 from an Amplicon");
	printf("\n\rDIO CT card.\n\r");
	printf("\n\r");

	if(argc != 4)
	{
		printusage();
		return 2;
	}

	// get commond line parameters
	cardtype = atoi(argv[1]);
	sscanf(argv[2], "%x", &address);
	irq = atoi(argv[3]);

	// register a suitable card
	hBoard = registerBoard((short)cardtype, (short)address, (short)irq);

	if (hBoard < 0)
	{
		printf("\n\rUnable to find Amplicon PC%dE DIO CT card at", cardtype);
		printf(" address %x irq %d", address, irq);
		return 1;
	}

	//
	// Set up PPI ports to mode 1 port A input, Port B output
	//
	DIOsetModeEx(hBoard, PPIX, M1_AIN_BOUT);  // 10110100

	DIOsetModeEx(hBoard, PPIX, ((7<<1)|1));  // set bit 7 (ack)
	DIOsetModeEx(hBoard, PPIX, ((6<<1)|1));  // set bit 6 (strobe)

	DIOsetModeEx(hBoard, PPIX, ENABLE_RIEA_INTE);  // 00001011  enable INTE A
	DIOsetModeEx(hBoard, PPIX, ENABLE_B_INTE);  // 00000101  enable INTE B

	hTCINTPortA = TCsetUserInterrupt( hBoard
					, &ReadProcessEvent
					, 0               	// user param passed to user fn as wParam
					, PPIX	+ 4			// PPIX C3 
					, ISR_READ_PPIC    // read port C must read data or we'll miss irqs
					, 0
					, 0
					, 0
					, 0
					);

	if (hTCINTPortA < 0)
	{
		printf("\n\r Can't acquire port C3 interrupt");
		FreeBoard(hBoard);
		return 1;
	}

	hTCINTPortB = TCsetUserInterrupt( hBoard
					, &WriteProcessEvent
					, 0            	// user param passed to user fn as wParam
					, PPIX 			// PPIX C0
					, ISR_READ_PPIC    // read port C must read data or we'll miss irqs
					, 0
					, 0
					, 0
					, 0
					);

	if (hTCINTPortB < 0)
	{
		printf("\n\r Can't acquire port C0 interrupt");
		TCfreeUserInterrupt(hBoard, hTCINTPortA);
		FreeBoard(hBoard);
		return 1;
	}

	printf("\n\rPress <return> to Stop");

	enableInterrupts( hBoard );

	// Yield remainder of timeslice to allow a user interrupt thread to start.
	Sleep(0);
	// Do it again for the other user interrupt thread, just in case.
	Sleep(0);

	// send one byte to kick xmitter
	start = clock();
	WriteProcessEvent(hBoard, 0, 0);

	going = TRUE;
	while(going)
	{
		if(_kbhit())
		{
			going = FALSE;
		}
		Sleep(500);
		printf("\rBytes transmitted = %lu, Bytes received = %lu, Errors = %lu", 
			ntxed, nrxed, nerrs);
	}

	disableInterrupts( hBoard );
    
	finish = clock();

	printf("\rBytes transmitted = %lu, Bytes received = %lu, Errors = %lu", 
		ntxed, nrxed, nerrs);

	duration = (double)(finish - start) / CLOCKS_PER_SEC;
	printf( "\r\n\r\n %.2f seconds elapsed\n", duration );

	printf("\r\nData transmission rate = %.1f bytes / sec", ntxed/duration);
	printf("\r\nData receive rate = %.1f bytes / sec", nrxed/duration);

	DIOsetModeEx(hBoard, PPIX, M0_AIN_BOUT);
	TCfreeUserInterrupt(hBoard, hTCINTPortB);
	TCfreeUserInterrupt(hBoard, hTCINTPortA);
	FreeBoard(hBoard);
	
	return 0;
}