//
// PC24 driver
//
// $Id$
//
// PROJECT DESCRIPTION
//
// Delphi 3 PC24 sample
//
// MODULE CONTENTS
//
// This example with driver the 4 PC24 DAC channels with
// user selectable wave forms and frequencies. It requires
// the card jumpers to be set so that Timer Counter 0
// generates an interrupt
//
// COPYRIGHT
//
// This source code is supplied as an example of interfacing 
// with an Amplicon PC24 DAC counter timer card in Delphi 3.0
// It may be freely re-used and modified, providing 
// relevant copyright is retained by MEV. Although the best endevours have 
// been made to ensure the accuracy of this code, MEV make no 
// warranty of any kind with regard to this sample source code.
//
// by MEV Ltd
// Copyright (c) 1997 MEV Limited
// Suite 4 Baxall Business Centre, Unit 25b,
// Adswood Road Ind Est, STOCKPORT, Cheshire. SK3 8LF
// +44 (0)161 477 1898
// @mev-ltd.demon.co.uk
//
// ENVIRONMENT
//    Compiler:      Borland Delphi 3 ( 32 Bit )
//    Target system: PC, running MS-Windows NT
//
// AMENDMENT RECORD
//
// $Log$
// Start   1.00  18/12/97
//
unit pc24unit;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, ExtCtrls, AmpComm;
  
type
  TDIO_BLKPORTIO = record
    Offset:DWORD;         // Port Offset in IO space
    length:DWORD;         // number of data bytes to write to consequetive ports
    Value: Array[0..3] of smallint;
  end;

  TFormPC24 = class(TForm)
    ButtonStart: TButton;
    RGChan1: TRadioGroup;
    RGChan2: TRadioGroup;
    RGChan3: TRadioGroup;
    RGChan4: TRadioGroup;
    ButtonStop: TButton;
    Label1: TLabel;
    LabelFound: TLabel;
    ButtonExit: TButton;
    Shape1: TShape;
    ButtonTest: TButton;
    RGFreq1: TRadioGroup;
    RGFreq2: TRadioGroup;
    RGfreq3: TRadioGroup;
    RGFreq4: TRadioGroup;
    RGAPI: TRadioGroup;
    CBIRQFreq: TComboBox;
    Label2: TLabel;
    LabelIRQFreq: TLabel;
    CBUpdate: TCheckBox;
    procedure ButtonStartClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure ButtonExitClick(Sender: TObject);
    procedure ButtonStopClick(Sender: TObject);
    procedure ButtonTestClick(Sender: TObject);
    procedure CBIRQFreqChange(Sender: TObject);
    procedure RGFreq1Click(Sender: TObject);
    procedure RGFreq2Click(Sender: TObject);
    procedure RGfreq3Click(Sender: TObject);
    procedure RGFreq4Click(Sender: TObject);
    procedure RGChan1Click(Sender: TObject);
    procedure RGChan2Click(Sender: TObject);
    procedure RGChan3Click(Sender: TObject);
    procedure RGChan4Click(Sender: TObject);
  private
    { Private declarations }
    function FindCard(cardtype:SmallInt):smallint;
  public                  
    { Public declarations }
  end; // TFormPC24 = class(TForm)
  
var
  FormPC24: TFormPC24;
  rho:real;
  theta:real;
  sigma:real;
  hBoard1:integer;
  hTCINT:integer;
  hADCINT:integer;

  ChanFn:Array [0..3] of smallint;
  ChanFreq:Array [0..3] of smallint;
  InPC24Event : Boolean;
  LastTime:TTimeStamp;
  IRQCount:Smallint;
  

procedure PC24ProcessEvent(h:smallint; wParam:WPARAM; Data:DWORD);stdcall;export;
function DoAnaloguefn(i, chan:smallint; theta:real):smallint;stdcall;export;
              
implementation

{$R *.DFM}

//-------------------------------------------------------------------------
// DoAnaloguefn
//-------------------------------------------------------------------------
// Function
//    Do an analogue fn based on index
//    Called by the interrupt callback fn for each channel
// Parameters
// Returns
//    - 
// Coded
//    helen                          19/12/97  
//-------------------------------------------------------------------------

function DoAnaloguefn(i, chan:smallint; theta:real):smallint;
var 
    value:smallint;
begin // DoAnaloguefn
    // make up some waveforms
    case i of 
    0: value := round(sin(theta) * $7ff) + $7FF;
    1: value := round(sin(theta + (PI/2)) * $7ff) + $7FF;
    2: value := round(sin(theta + (PI)) * $7ff) + $7FF;
    3: value := round(sin(theta+ (3*PI/2)) * $7ff) + $7FF;
    4: Value := $FFF * round(theta / (2*PI));     
    5: Value := round($7FF * ((theta / (PI) - 1))) + $7F0;
    else         
    begin
       value := 0;
    end;       
    end; // case i of

    Result := value;
end;     // DoAnaloguefn

//-------------------------------------------------------------------------
// PC24ProcessEvent
//-------------------------------------------------------------------------
// Function
//    This fn is called by DIO_TC when an PC24 timer interrupt
//    occurs. Its is prototyped as export so that the stack is
//    set up properly.
// Parameters
// Returns
//    - 
// Coded
//    helen                          19/12/97  
//-------------------------------------------------------------------------
procedure PC24ProcessEvent(h:smallint; wParam:WPARAM; Data:DWORD);
var
   i:integer;
   value: smallint;
   IOBuffer:TDIO_BLKPORTIO;
   TimeNow:TTimeSTamp;
   Diff:DWORD;
   f:real;
begin // PC24ProcessEvent
    if InPC24Event then exit;
    InPC24Event := true;

    if FormPC24.CBUpdate.Checked Then
    begin
       inc(IRQCount);

       // work out real IRQ freq
       if IRQCount > 1000 then
       begin
           IRQCount := 0;
           TimeNow := DateTimeToTimeStamp(now());
           DIFF := TimeNow.Time -  LastTime.Time;
           LastTime := TimeNow;

           try
              f := 1000000/ DIFF;
           except
           on EZeroDivide do f:= 0;
           end;
           
           FormPC24.LabelIRQFreq.Caption := FormatFloat('###0', f) + ' Hz';
       
           if FormPC24.Shape1.brush.color = clRed then
              FormPC24.Shape1.brush.color := clGray
           else           
              FormPC24.Shape1.brush.color := clRed;
       end;
    end;
        
    // roll round number between 0 and 2 PI
    theta := (theta + ((20/4096) * 2 * PI));
    if ( theta >= (2*PI)) then theta := theta - (2*PI);

    rho := (rho + ((40/4096) * 2 * PI));
    if ( rho >= (2*PI)) then rho := rho - (2*PI);

    sigma := (sigma + ((60/4096) * 2 * PI));
    if ( sigma >= (2*PI)) then sigma := sigma - (2*PI);

    // generate pretty wave forms
    if FormPC24.RGAPI.ItemIndex = 0 Then
    begin
        // API uses and IOCTL call for each channel
        // there is a latency associated with wach IOCTL
        // this slows down our ability to process interrupts
        for i := 0 to 3 do
        begin
            case ChanFreq[i] of
            1: Value := DoAnaloguefn(Chanfn[i], i, theta);
            2: Value := DoAnaloguefn(Chanfn[i], i, rho);
            3: Value := DoAnaloguefn(Chanfn[i], i, sigma);
            else
                Value := 0;
            end; // case
        PC24setData(hboard1, i, value);  // 70uS 
        end;

        // max IRQ freq = 70us + (4*70us) => 2857 hz
        
    end
    else
    begin
        // there is a latency associated with wach IOCTL
        // use direct IOCTL to set all four channels at once
        // IRQ serviced at higher speed
        IOBuffer.Offset := 0;
        IOBuffer.Length := 8; // bytes

        for i := 0 to 3 do
        begin
            case ChanFreq[i] of
            1: IOBuffer.Value[i] := DoAnaloguefn(Chanfn[i], i, theta);
            2: IOBuffer.Value[i] := DoAnaloguefn(Chanfn[i], i, rho);
            3: IOBuffer.Value[i] := DoAnaloguefn(Chanfn[i], i, sigma);
            end; // case
        end;

        DIO_TC_IOCTL(hBoard1, IOCTL_SET_IODATA, @IOBuffer, sizeof(TDIO_BLKPORTIO)); // 70us

        // max IRQ freq = 70us + (70us) => 7142 hz
        
    end;
    InPC24Event := false;
                                         
end; // PC24ProcessEvent

//-------------------------------------------------------------------------
// FindCard
//-------------------------------------------------------------------------
// Function
//    register a board of a particular type
//    uses new extended register board function that doesn't
//    need to know card address etc
// Parameters
// cardtype     :SmallInt
// Returns
//    - 
// Coded
//    helen                          19/12/97  
//-------------------------------------------------------------------------
function TFormPC24.FindCard(cardtype:SmallInt):smallint;
var 
    i:integer;
    hboard:smallint;
begin // FormCreate

     // find a board using registerBoardEx which doesn't
     // need to know base address etc etc
     i := 0;
     repeat  
          hboard := registerBoardEx(i);

          if hboard = OK then
          begin
               // if its a PC24 then proceed
               if GetBoardModel(hboard) = cardtype Then 
               begin
                  Result := hBoard;
                  exit;
               end;                  
          end;
          FreeBoard(hBoard);
          inc(i);
     until (i >= 8);          

     Result := -1;     
end;
     
//-------------------------------------------------------------------------
// FormCreate
//-------------------------------------------------------------------------
// Function
//    register board(s) and get ready to start
// Parameters
// Returns
//    - 
// Coded
//    helen                          19/12/97  
//-------------------------------------------------------------------------

procedure TFormPC24.FormCreate(Sender: TObject);
var
   i:integer;
begin // FormCreate

     // no handles
     hTCINT := -1;
     hADCINT := -1;
     hboard1 := -1;
     InPC24Event := FALSE;
  
     CBIrqFreq.ItemIndex := 0;

     // set up IRQ vars
     IRQCount := 0;
     for i := 0 to 3 do
     begin
         ChanFn[i] := 6;
         ChanFreq[i] := 1;
     end;

     // start rolling count at zero;
     rho := 0;
     theta := 0;
     sigma := 0;
          
     // find a board using registerBoardEx which doesn't
     // need to know base address etc etc
     hBoard1 := FindCard(PC24);
     
     // if we found a suitable card. enable buttons     
     if hBoard1 >=0 then
     begin
          ButtonStart.Enabled := True;
          ButtonTest.Enabled := True;
          LabelFound.Caption := 'Found PC24 card';
          LabelFound.Font.Color := clGreen;
     end                   
     else
     begin
          LabelFound.Caption := 'PC24 card not found';
          LabelFound.Font.Color := clRed;
     end;                                   

     LastTime := DateTimeToTimeStamp(now())
end; // FormCreate

//-------------------------------------------------------------------------
// FormClose
//-------------------------------------------------------------------------
// Function
//    Program closing free any resources
// Parameters
//    Sender : TObject                           
//    var Action : TCloseAction                  
// Returns
//    - 
// Coded
//    helen                          19/12/97  
//-------------------------------------------------------------------------

procedure TFormPC24.FormClose(Sender: TObject; var Action: TCloseAction);
begin // FormClose
     // if we have handles, free resource
     if hTCINT >= 0 Then TCfreeUserInterrupt (hboard1, hTCINT);
     if hboard1 >= 0 Then FreeBoard(hboard1);
end; // FormClose


//-------------------------------------------------------------------------
// ButtonStartClick
//-------------------------------------------------------------------------
// Function
//    Start Timer interrupt to generate waveform
//    set up sample interrupt if PC27 fitted
// Parameters
// Returns
//    - 
// Coded
// Coded
//    helen                          19/12/97  
//-------------------------------------------------------------------------
procedure TFormPC24.ButtonStartClick(Sender: TObject);
var
   ret:smallint;
begin // ButtonStartClick

   // get resonable sample clock, PC24 interrupts are
   // generated at this rate
   // writing to 4 word DAC channels takes 8 IOCTLs
   // IOCTL latency is 70us hence max freq for 4 chans is 1785 hz
	ret := TCgenerateFreq(hboard1, Y1, 0, (CBIRQFreq.ItemIndex+1) * 1000); // hz
   
	if(ret <> OK) then
	begin
		messagedlg('Cant find counter timer Y1', mtinformation, [mbok], 0);
		exit;
	end;
      
     // Set up an interrupt call back on timer counter 0 interrupt
     //
     hTCINT := TCsetUserInterrupt( hboard1
                                 , @PC24ProcessEvent
                                 , 0			// user param passed to user fn as wParam
                                 , Y1				// IRQ on Y1 timer
                                 , ISR_NODATA
                                 , 0
                                 , 0
                                 , 0
                                 , 0
                                 );

      // if we have the interrupt OK, lets go, ps enable stop buttons     
      if hTCINT >= 0 Then
      begin
          enableInterrupts( hboard1 );
          ButtonStop.Visible := True;
          ButtonStart.Visible := False;
      end
      else
          MessageDlg('Unable to assign IRQ', mtInformation, [mbOK],0);
     
end; // ButtonStartClick

//-------------------------------------------------------------------------
// ButtonStopClick
//-------------------------------------------------------------------------
// Function
//   Stop generating wave forms
// Parameters
// Returns
//    - 
// Coded
//    helen                          19/12/97  
//-------------------------------------------------------------------------

procedure TFormPC24.ButtonStopClick(Sender: TObject);
begin // ButtonStopClick
     // stop 
     disableInterrupts(hboard1); 
     TCfreeUserInterrupt (hboard1, hTCINT);
     // enable go buttons
     hTCINT := -1;             
     ButtonStop.Visible := False;
     ButtonStart.Visible := True;
end; // ButtonStopClick

//-------------------------------------------------------------------------
// CBIRQFreqChange
//-------------------------------------------------------------------------
// Function
//    change frequency
// Parameters
// Returns
//    - 
// Coded
//    helen                          19/12/97  
//-------------------------------------------------------------------------

procedure TFormPC24.CBIRQFreqChange(Sender: TObject);
begin
    TCgenerateFreq(hboard1, Y1, 0, (CBIRQFreq.ItemIndex+1) * 1000); // hz
end;

//-------------------------------------------------------------------------
// ButtonTestClick
//-------------------------------------------------------------------------
// Function
//    Slowly ramps DAC channels
// Parameters
// Returns
//    - 
// Coded
//    helen                          19/12/97  
//-------------------------------------------------------------------------

procedure TFormPC24.ButtonTestClick(Sender: TObject);
var
   i : integer;
begin // ButtonTestClick
     ButtonTest.Enabled := false;
     for i := 0 to 50 do
     begin // for i :=..
         PC24setData(hboard1, 0, i*4095 div 50);
         PC24setData(hboard1, 1, i*4095 div 50);
         PC24setData(hboard1, 2, i*4095 div 50);
         PC24setData(hboard1, 3, i*4095 div 50);
         sleep(50);
    end; // for i :=..
    ButtonTest.Enabled := true;
end; // ButtonTestClick

//-------------------------------------------------------------------------
// ButtonExitClick
//-------------------------------------------------------------------------
// Function
//    Exit the program
// Parameters
// Returns
//    - 
// Coded
//    helen                          19/12/97  
//-------------------------------------------------------------------------

procedure TFormPC24.ButtonExitClick(Sender: TObject);
begin // ButtonExitClick
     close;
end; // ButtonExitClick

//-------------------------------------------------------------------------
// RGFreq Click
//-------------------------------------------------------------------------
// Function
//   Set up freq arrays 
// Parameters
// Returns
//    - 
// Coded
//    helen                          19/12/97  
//-------------------------------------------------------------------------

procedure TFormPC24.RGFreq1Click(Sender: TObject);
begin
    ChanFreq[0] :=  RGFreq1.ItemIndex + 1;
end;

procedure TFormPC24.RGFreq2Click(Sender: TObject);
begin
    ChanFreq[1] :=  RGFreq2.ItemIndex + 1;
end;

procedure TFormPC24.RGfreq3Click(Sender: TObject);
begin
    ChanFreq[2] :=  RGFreq3.ItemIndex + 1;
end;

procedure TFormPC24.RGFreq4Click(Sender: TObject);
begin
    ChanFreq[3] :=  RGFreq4.ItemIndex + 1;
end;

//-------------------------------------------------------------------------
// RGChan fn Click
//-------------------------------------------------------------------------
// Function
//   Set up chan fn arrays 
// Parameters
// Returns
//    - 
// Coded
//    helen                          19/12/97  
//-------------------------------------------------------------------------
procedure TFormPC24.RGChan1Click(Sender: TObject);
begin
    ChanFn[0] :=  RGChan1.ItemIndex;
end;

procedure TFormPC24.RGChan2Click(Sender: TObject);
begin
    ChanFn[1] :=  RGChan2.ItemIndex;
end;

procedure TFormPC24.RGChan3Click(Sender: TObject);
begin
    ChanFn[2] :=  RGChan3.ItemIndex;
end;

procedure TFormPC24.RGChan4Click(Sender: TObject);
begin
    ChanFn[3] :=  RGChan4.ItemIndex;
end;

end.
