{
  (C) 2000
  MEV System Design Consultancy
   
  This software is provided 'as-is', without any express or implied
  warranty.  In no event will the authors be held liable for any damages
  arising from the use of this software.

  Permission is granted to anyone to use this software for any purpose,
  including commercial applications, and to alter it and redistribute it
  freely, subject to the following restrictions:

  1. The origin of this software must not be misrepresented; you must not
     claim that you wrote the original software. If you use this software
     in a product, an acknowledgment in the product documentation would be
     appreciated but is not required.
  2. Altered source versions must be plainly marked as such, and must not be
     misrepresented as being the original software.
  3. This notice may not be removed or altered from any source distribution.

  Web   - mev.co.uk
  email - keith@mev.co.uk

  Modified 01/06/2000 - IJA
    Invalidate last mouse coordinates when setting explicit knob value.
    Added a MouseDwn event.
    Reduced code to work out angle from mouse position.
    Fixed assumptions about MinDeg and MaxDeg constants.

}
unit Knob;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,Math;

type
  TKnob = class(TGraphicControl)
  private
    { Private declarations }
    FBrush :    TBrush;
    FPen   :    TPen;
    FOnChange : TNotifyEvent;
    FKnobValue : Integer;

    KnobVal :     Integer;
    LastKnobVal : Integer;
    KnobCanvas :  TCanvas;
    ForceDraw :   Boolean;
    Angle :       Real;
    LastAngle :   Real;
    MouseX :      Integer;
    MouseY :      Integer;

    Procedure   SetBrush( Value: TBrush );
    Procedure   SetPen( Value: TPen );
    function    GetValue: Integer;
    Procedure   SetKnobValue( Value: Integer );
    procedure   DrawSmallCircle;
    Procedure   DrawFromValue( Setting: Integer );
  protected               
    { Protected declarations }
    procedure   paint; override;
    procedure   MouseMove( Shift: TShiftState; X, Y: Integer ); override;
    procedure   MouseDown( Button: TMouseButton; Shift: TShiftState; X, Y: Integer ); override;
  public
    { Public declarations }
    constructor Create( AOwner: TComponent ); override;
    destructor  Destroy; override;
    procedure   PaintToCanvas( CanvasRect: TRect );
  public
    { Public declarations }
  published
    { Published declarations }
    Property KnobValue : Integer Read FKnobValue write SetKnobValue;
    Property Align;
    Property OnChange : TNotifyEvent read FOnChange write FOnChange;
    Property OnMouseDown;
    Property OnMouseUp;
    Property Height default 50;
    Property Width  default 50;
    Property Brush: TBrush  read FBrush write SetBrush;
    Property Pen:   TPen    read FPen   write SetPen;
    Property Value: Integer read GetValue;
    procedure StyleChanged( Sender: TObject );
  end;
  Const
    MinDeg   =  30.00;
    StartDeg =  30.00;
    MaxDeg   = 330.00;
                        
procedure Register;

implementation
//-------------------------------------------------------------------------
// Constructor
//-------------------------------------------------------------------------
// Function
//   Creates pen, brush etc belonging to Knob
// Parameters
//   Owner   
// Returns
//   None
//-------------------------------------------------------------------------
constructor TKnob.Create( AOwner: TComponent );
begin
  inherited Create( AOwner );
  FBrush          := TBrush.Create;
  FPen            := TPen.Create;
  FBrush.OnChange := StyleChanged;
  // Set Default Colours
  FBrush.Color    := clSilver;
  FPen.Color      := clBlack;
  Height          := 50;
  Width           := 50;
  KnobVal         := 0;
  LastKnobVal     := KnobVal;
  ForceDraw       := FALSE;
  Angle           := StartDeg;
  LastAngle       := Angle;
  MouseX          := -1;
  MouseY          := -1;

end;

//-------------------------------------------------------------------------
// Destructor
//-------------------------------------------------------------------------
// Function
//    Destroys pen, brush etc belonging to Knob
// Parameters
//    Owner
// Returns
//   None
//-------------------------------------------------------------------------
destructor TKnob.Destroy;
begin
  FPen.Free;
  FBrush.Free;
  inherited destroy();
end;

Procedure TKnob.SetKnobValue( Value: Integer );
var
   V: integer;
begin
   V := Value;
   if Value < 0 then V := 0;
   if Value > 10000 then V := 10000;
   DrawFromValue( V );
   MouseX := -1;
   MouseY := -1;
   ForceDraw := TRUE;
   Invalidate;
end;

//-------------------------------------------------------------------------
// StyleChanged
//-------------------------------------------------------------------------
// Function
//   Associated with pen, brush on change this routine
//   causes a repaint if these objects properties change
// Parameters
//   Sender
// Returns
//   None
//-------------------------------------------------------------------------
procedure TKnob.StyleChanged( Sender: TObject );
begin
  Invalidate;
end;

//-------------------------------------------------------------------------
// SetBrush
//-------------------------------------------------------------------------
// Function
//   Sets the brush used for the background
// Parameters
//   value the brush
// Returns
//   None
//-------------------------------------------------------------------------
Procedure TKnob.SetBrush( Value: TBrush );
begin
  FBrush.Assign( Value );
end;                    

//-------------------------------------------------------------------------
// SetPen
//-------------------------------------------------------------------------
// Function
//    sets the pen used for the cursors
// Parameters
//   value the Pen
// Returns
//   None
//-------------------------------------------------------------------------
Procedure TKnob.SetPen( Value: TPen );
begin
  FPen.Assign( Value );
  Invalidate;
end;                    

//-------------------------------------------------------------------------
// GetValue
//-------------------------------------------------------------------------
// Function
// Parameters                                             
//   value the Knob Value 0 to 10000
// Returns
//   None
//-------------------------------------------------------------------------
function TKnob.GetValue: Integer;
begin
  result := KnobVal;
end;

//-------------------------------------------------------------------------
// MouseMove
//-------------------------------------------------------------------------
// Function
//    Check that the Mouse is ON the KNOB
// Parameters
//    Shift state
//    Mouse Coords
// Returns
//   None
//-------------------------------------------------------------------------
procedure TKnob.MouseMove( Shift: TShiftState; X, Y: Integer );
begin
  inherited MouseMove( Shift, X, Y );
  if ssLeft in Shift then { make sure button is down }
  begin
    MouseX := -1;
    MouseY := -1;
    If ( X > 0 ) and ( X < Width ) and (Y > 0) and (Y < Height) then
    begin
      MouseX := X;
      MouseY := Y;
      invalidate;
    end;
  end;
end;

//-------------------------------------------------------------------------
// MouseDown
//-------------------------------------------------------------------------
// Function
//    Check that the Mouse is ON the KNOB
// Parameters
//    Button
//    Shift state
//    Mouse Coords
// Returns
//   None
//-------------------------------------------------------------------------
procedure TKnob.MouseDown( Button: TMouseButton; Shift: TShiftState; X, Y: Integer );
begin
  inherited MouseDown( Button, Shift, X, Y );
  MouseMove(Shift, X, Y);
end;

//-------------------------------------------------------------------------
// Paint
//-------------------------------------------------------------------------
// Function
//    Paints on the supplied canvas
// Parameters
//    Sender
// Returns
//   None
//-------------------------------------------------------------------------
procedure TKnob.Paint;
var
  CanvasRect: TRect;
begin
  with Canvas do
  begin
    KnobCanvas := Canvas;
    // Set up drawing implements
    Brush := FBrush;
    Pen   := FPen;
    // sort out our peice of paper
    CanvasRect.Left   := 0;
    CanvasRect.Top    := 0;
    CanvasRect.Right  := Width - 5;
    CanvasRect.Bottom := Height;
    // Blank the paper                   
    Fillrect( CanvasRect );
    PaintToCanvas( CanvasRect );
  end;
end;

//-------------------------------------------------------------------------
// PaintToCanvas
//-------------------------------------------------------------------------
// Function
//    Draw the graph on the controls canvas
// Parameters
//   TheCanvas    The paper
//   CanvasRect   The area we can draw in
// Returns
//   None
//-------------------------------------------------------------------------

procedure TKnob.PaintToCanvas( CanvasRect: TRect );
Var
  XC :           Real;
  YC :           Real;
  H :            Real;
  GoodMove :     Boolean;

  C1Width, C1Offset : Integer;
begin
  // Keep it Square
  if Width > Height then
    Height := Width;
  if Height > Width then
    Width := Height;

  // Find Center of Control
  XC := Width / 2;
  YC := Height / 2;

  C1Width :=  ( width * 80 ) DIV 100; // 80 %
  C1Offset :=  ( Width - C1Width ) DIV 2;
  KnobCanvas.ellipse( C1Offset, C1Offset, C1Width + C1Offset, C1Width + C1Offset );

  // Make Sure within the Control
  if ( ( MouseX >= 0 ) and ( MouseY >= 0 ) ) or ( ForceDraw ) then
  begin
    H := sqrt( sqr( XC - MouseX ) + sqr( YC - MouseY ) );
    // Check that te Mouse is IN the Circle & not in the Center of the Circle
    if ( ( H < ( Width DIV 2 ) ) and ( H > Width DIV 8 ) ) or ( ForceDraw ) then
    begin
      if NOT ( ForceDraw ) then
      begin
        //
        // Angle (in degrees from 0 to 360) increases clockwise from bottom.
        //
        //    (0,0)-----------------> xaxis
        //     |          180
        //     |
        //     |
        //     |
        //     | 90     (XC,YC)    270
        //     |         ,':
        //     |       ,'  :
        //     |     ,'<Ang:
        //     |  (X,Y)    :
        //     V           0
        //   yaxis
        //
        if MouseY = YC then
        begin
          //
          // SpecialCase Exactly on the Hoz Line thru centre
          // ( Stops DIV by Zero )
          //
          if MouseX < XC then
            Angle := 90
          else
            Angle := 270;
        end
        else
        begin
          Angle := RadToDeg( arctan ( (XC - MouseX) / (MouseY - YC) ) );
          if MouseY < YC then
            Angle := Angle + 180
          else
            if Angle < 0 then Angle := Angle + 360;
        end;
      end;

      if ForceDraw then
      begin
        LastAngle := Angle;
      end;

      // Stop Jumping through 0
      GoodMove := ( abs( Angle - LastAngle ) <= 90 );
      if NOT GoodMove then Angle := LastAngle;

      // Limit it to Selected Limits
      if Angle < MinDeg then
        Angle := MinDeg;
      if Angle > MaxDeg then
        Angle := MaxDeg;

      LastAngle := Angle;
    end  // of In Circle
    else
    begin
      Angle := LastAngle;  // Not In Circle so use last Angle
    end;
  end
  else
    Angle := LastAngle;     // Not In Control so use Last Angle

  // Do Small Circle
  DrawSmallCircle;

  KnobVal := Round( ( ( Angle - MinDeg ) * 10000 ) / ( MaxDeg - MinDeg ) );
  if KnobVal <> LastKnobVal then
  Begin
    LastKnobVal := KnobVal;
    if Assigned( FOnCHange ) then
      FOnChange( Self );
  end;
  ForceDraw := FALSE;
    
end;

procedure TKnob.DrawSmallCircle;
var
  R : Real;
  CX, CY: Real;
  C1Width, C2Width, C2Left, C2Top : Integer; 
Begin
  C1Width :=  ( width * 80 ) DIV 100; // 80 %
  // Do Small Circle  
  C2Width :=  ( width * 10 ) DIV 100; // 10 %
  R := ( C1Width DIV 2 ) - (  Width * 10  DIV 100 ); // - 10%
  CX := ( R * Sin( DegToRad( Angle ) ) ) * - 1 + R;
  CY := R * Cos( DegToRad( Angle ) ) + R;
  C2Left := Trunc( CX ) + ( Width * 15  DIV 100 ); // + 15%
  C2Top  := Trunc( CY ) + ( Width * 15  DIV 100 ); // + 15%

  KnobCanvas.ellipse( C2Left, C2Top, C2Width + C2Left, C2Width + C2Top );
end;

Procedure TKnob.DrawFromValue( Setting: Integer );
var
  R : Real;
  CX, CY: Real;
  C1Width, C2Width, C2Left, C2Top : Integer; 
begin
  if ( Width <> 0 ) and ( KnobCanvas <> Nil ) then
  begin
    C1Width :=  ( width * 80 ) DIV 100; // 80 %
    // Do Small Circle
    C2Width :=  ( width * 10 ) DIV 100; // 10 %
    R := ( C1Width DIV 2 ) - (  Width * 10  DIV 100 ); // - 10%
    // Globally Set the Angle as this will be an override of the Knob
    Angle := ( ( Setting / 10000 ) * (MaxDeg - MinDeg) ) + MinDeg;
    LastAngle := Angle; //??
    CX := ( R * Sin( DegToRad( Angle ) ) ) * - 1 + R;
    CY := R * Cos( DegToRad( Angle ) ) + R;
    C2Left := Trunc( CX ) + ( Width * 15  DIV 100 ); // + 15%
    C2Top  := Trunc( CY ) + ( Width * 15  DIV 100 ); // + 15%

    KnobCanvas.ellipse( C2Left, C2Top, C2Width + C2Left, C2Width + C2Top );
  end;
end;

procedure Register;
begin
  RegisterComponents('MEV', [TKnob]);
end;

end.


